CREATE OR REPLACE PACKAGE SIM2_IMPORT_PKG  IS

   /* import utils by SNG Feb 2000 */  

   TYPE col_table_type IS TABLE OF VARCHAR2(2000) INDEX BY BINARY_INTEGER;
   
   TYPE t_activity_count IS TABLE OF NUMBER(6) INDEX BY BINARY_INTEGER;       
   
   v_activity_count     t_activity_count; 
  
   PROCEDURE parse_line_table ( pv_line IN VARCHAR2, p_col_table OUT col_table_type, p_no_cols IN NUMBER, 
                                p_delimiter IN VARCHAR2 DEFAULT ',', p_bad_parse IN OUT BOOLEAN,
                                p_import_log_seq IN NUMBER);

   FUNCTION get_one_col (pv_line IN VARCHAR2, p_col_no IN NUMBER, p_delimiter IN VARCHAR2,
                         p_import_log_seq IN NUMBER) RETURN VARCHAR2;

   PROCEDURE WRITE_ERROR (p_import_log_seq IN NUMBER, p_desc IN VARCHAR2, p_sta_seq IN NUMBER,
                          p_fa_seq IN NUMBER, p_proj_seq IN NUMBER);

   PROCEDURE DELETE_IMPORT (p_import_seq IN NUMBER);
 
   FUNCTION GET_IMPORT_LOG_SEQ RETURN NUMBER;
                            
   PROCEDURE UPDATE_SIL_ROWS_READY(p_import_log_seq IN NUMBER, p_rows_ready_change IN NUMBER);
   
   PROCEDURE UPDATE_SIL_ROWS_ERRORS(p_import_log_seq IN NUMBER, p_rows_errors_change IN NUMBER);
   
   PROCEDURE UPDATE_SIL_ROWS_IMPORTED(p_import_log_seq IN NUMBER, p_rows_imported_change IN NUMBER);
                     
                                                    
END;
/
CREATE OR REPLACE PACKAGE BODY SIM2_IMPORT_PKG as

/******************************************************************************************************/

  /* PROC TO PARSE one comma delimited line into a table */

  PROCEDURE parse_line_table (pv_line IN VARCHAR2, p_col_table OUT col_table_type, p_no_cols IN NUMBER,
                              p_delimiter IN VARCHAR2 DEFAULT ',', p_bad_parse IN OUT BOOLEAN,
                              p_import_log_seq IN NUMBER) IS

     v_col_cnt BINARY_INTEGER := 1;

  BEGIN
  
     WHILE v_col_cnt <= p_no_cols  LOOP
         
         p_col_table(v_col_cnt) := get_one_col(pv_line, v_col_cnt, p_delimiter,p_import_log_seq);

         v_col_cnt := v_col_cnt + 1;
     END LOOP;

     IF v_col_cnt - 1 != p_no_cols THEN
       p_bad_parse := TRUE;
     END IF;
     
  EXCEPTION
     WHEN OTHERS THEN
        sim2_import_pkg.write_error(p_import_log_seq, SQLERRM||'- in parse_line_table function, column '||p_no_cols,null,null,null);
     
  END parse_line_table;


/******************************************************************************************************/

  /* Internal function to get each column - used by parse_line_table proc */

  FUNCTION get_one_col (pv_line IN VARCHAR2, p_col_no IN NUMBER, p_delimiter IN VARCHAR2,
                        p_import_log_seq IN NUMBER) RETURN VARCHAR2 IS

     v_first_comma NUMBER;
     v_start_pos   NUMBER;

     v_col_cnt     NUMBER := 1;
     v_line_length  NUMBER;
     v_char_cnt     BINARY_INTEGER := 1;

     BEGIN


       v_line_length := LENGTH(pv_line);
       WHILE v_char_cnt <= v_line_length LOOP
           IF SUBSTR(pv_line,v_char_cnt,1) = p_delimiter THEN
              v_col_cnt := v_col_cnt + 1;
           END IF;
           v_char_cnt := v_char_cnt + 1;
       END LOOP; 


     IF p_col_no > v_col_cnt THEN
 
        RETURN NULL;

     ELSE
     
        IF p_col_no = 1 THEN
          v_start_pos := 1;
        ELSE
          v_start_pos := INSTR(pv_line,p_delimiter ,1,p_col_no-1) + 1;
        END IF;

        v_first_comma := INSTR(pv_line,p_delimiter ,1,p_col_no);
        IF v_first_comma = 0  THEN
           v_first_comma := v_line_length + 1 ;
        END IF;
 
        RETURN SUBSTR(pv_line,v_start_pos,v_first_comma - v_start_pos );

     END IF;
  EXCEPTION 
    WHEN OTHERS THEN
      sim2_import_pkg.write_error(p_import_log_seq, SQLERRM||'- in get_one_col function, column '||p_col_no,null,null,null);
     
  END get_one_col;


/******************************************************************************************************/
PROCEDURE WRITE_ERROR (p_import_log_seq IN NUMBER, p_desc IN VARCHAR2, p_sta_seq IN NUMBER,
                       p_fa_seq IN NUMBER, p_proj_seq IN NUMBER) IS

BEGIN


    INSERT INTO SIM_IMPORT_DETAILS ( 
      SID_SEQ,
      SID_SIL_SEQ, 
      SID_DESC,
      SID_STA_SEQ,
      SID_FA_SEQ,
      SID_PROJ_SEQ
      ) 
    VALUES (
      SIM_IMPORT_DETAILS_SEQ.NEXTVAL,
      p_import_log_seq, 
      p_desc,
      p_sta_seq,
      p_fa_seq,
      p_proj_seq
      );

    COMMIT;


END WRITE_ERROR;


/******************************************************************************************************/
PROCEDURE DELETE_IMPORT (p_import_seq IN NUMBER) IS

   v_delete_cnt          NUMBER(10) := 1;
   v_well_delete         NUMBER(10) := 1;
   v_import_type         VARCHAR2(30);
   
   v_created_by          NUMBER(1);
   v_visit_is_nbr        NUMBER(8);
   v_visit_org_id        VARCHAR2(8);
   v_trip_is_nbr         NUMBER(8);
   v_trip_org_id         VARCHAR2(8);
   v_dummy_seq           NUMBER(10);
   v_visit_flag          CHAR(1);
   v_trip_flag           CHAR(1);
   
   
   CURSOR c_import_type (p_import_seq IN NUMBER) IS
      SELECT SID_TYPE FROM SIM_IMPORT_LOG, SIM_IMP_CFG, SIM_IMP_DEFS
        WHERE sil_seq = p_import_seq
          AND sil_sicc_seq = sicc_seq
          AND sicc_sid_seq = sid_seq;
   
   CURSOR c_field_activities (p_import_seq IN NUMBER) IS
      SELECT fa_seq, fa_ssv_seq, fa_stp_seq FROM SIM_FIELD_ACTIVITIES WHERE fa_import_seq = p_import_seq;
/*
   CURSOR c_get_visit_info(p_visit_seq IN NUMBER) IS
     SELECT SSV_CREATED_BY_SIM, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID 
       FROM SIM_STATION_VISITS WHERE SSV_SEQ = p_visit_seq;
*/
   CURSOR c_get_visit_info(p_visit_seq IN NUMBER) IS
     SELECT ssv_seq -- SSV_CREATED_BY_SIM, SSV_TSRSTVST_IS_NUMBER, SSV_TSRSTVST_ORG_ID 
       FROM SIM_STATION_VISITS SSV, TSRSTVST T
       WHERE SSV.SSV_TSRSTVST_IS_NUMBER = T.TSRSTVST_IS_NUMBER
       AND   SSV.SSV_TSRSTVST_ORG_ID    = T.TSRSTVST_ORG_ID
       AND   SSV.SSV_SEQ = p_visit_seq;
/*
   CURSOR c_get_trip_info(p_trip_seq IN NUMBER) IS
     SELECT STP_CREATED_BY_SIM, STP_TSRTRIP_IS_NUMBER, STP_TSRTRIP_ORG_ID 
       FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;       
*/   
   CURSOR c_get_trip_info(p_trip_seq IN NUMBER) IS
     SELECT stp_seq -- STP_CREATED_BY_SIM, STP_TSRTRIP_IS_NUMBER, STP_TSRTRIP_ORG_ID 
       FROM SIM_TRIPS ST, TSRTRIP T
       WHERE ST.STP_TSRTRIP_IS_NUMBER = T.TSRTRIP_IS_NUMBER
       AND   ST.STP_TSRTRIP_ORG_ID    = T.TSRTRIP_ORG_ID
       AND   STP_SEQ = p_trip_seq;

   CURSOR c_check_visit_in_SIM(p_visit_seq IN NUMBER) IS
     SELECT FA_SEQ FROM SIM_FIELD_ACTIVITIES WHERE FA_SSV_SEQ = p_visit_seq;
   
   CURSOR c_check_trip_in_SIM(p_trip_seq IN NUMBER) IS
     SELECT FA_SEQ FROM SIM_FIELD_ACTIVITIES WHERE FA_STP_SEQ = p_trip_seq;

   CURSOR c_stations (p_import_seq IN NUMBER) IS 
      SELECT sta_seq FROM SIM_STATIONS WHERE sta_import_seq = p_import_seq;   
 
   CURSOR c_wells (p_sta_seq IN NUMBER) IS 
      SELECT ssw_seq FROM SIM_STATION_WELLS WHERE ssw_sta_seq = p_sta_seq;   
       
   CURSOR c_acodes (p_import_seq IN NUMBER) IS
     SELECT ac_seq FROM SIM_ANALYSIS_CODES WHERE ac_import_log_seq = p_import_seq;
      
   CURSOR c_check_import_line(p_import_seq IN NUMBER) IS
     SELECT sipl_seq FROM SIM_IMP_LINES
       WHERE sipl_sil_seq = p_import_seq;

   CURSOR c_check_import_detail(p_import_seq IN NUMBER) IS
     SELECT sid_seq FROM SIM_IMPORT_DETAILS
       WHERE sid_sil_seq = p_import_seq;
       
BEGIN

   OPEN c_import_type(p_import_seq);
   FETCH c_import_type INTO v_import_type;
   CLOSE c_import_type;
   
   
   IF v_import_type = 'STATIONS' THEN
      /* DELETING A STATIONS IMPORT */
   
      FOR v_stations IN c_stations(p_import_seq) LOOP
         
         DELETE SIM_STATION_LOCATIONS WHERE STL_STA_SEQ = v_stations.sta_seq;
         DELETE SIM_STATIONS WHERE STA_SEQ = v_stations.sta_seq;
         
         IF v_delete_cnt > 100 THEN
            COMMIT;
            v_delete_cnt := 0;
         END IF;
      
         v_delete_cnt := v_delete_cnt + 2;
      
      END LOOP;
      
   ELSIF v_import_type = 'GW STATIONS' THEN
      /* DELETING A STATIONS IMPORT */
   
      FOR v_stations IN c_stations(p_import_seq) LOOP

         v_well_delete := 0; 
         FOR v_wells in c_wells(v_stations.sta_seq) LOOP
            DELETE SIM_STATION_CASING WHERE SSC_SSW_SEQ = v_wells.ssw_seq;
            DELETE SIM_STATION_INTVL WHERE SSI_SSW_SEQ = v_wells.ssw_seq;
            v_well_delete := v_well_delete + 2;
            IF v_well_delete > 100 THEN
               COMMIT;
               v_well_delete := 0;
            END IF;
      
         END LOOP;
         
         
         DELETE SIM_STATION_WELLS WHERE SSW_STA_SEQ = v_stations.sta_seq;
         DELETE SIM_STATION_LOCATIONS WHERE STL_STA_SEQ = v_stations.sta_seq;
         DELETE SIM_STATION_IGUA WHERE SIG_STA_SEQ = v_stations.sta_seq;
         DELETE SIM_STATIONS WHERE STA_SEQ = v_stations.sta_seq;
         
         IF v_delete_cnt > 100 THEN
            COMMIT;
            v_delete_cnt := 0;
         END IF;
      
        /* ESTIMATED DELETE COUNT*/
         v_delete_cnt := v_delete_cnt + 10;
      
      END LOOP;
   
   ELSIF v_import_type = 'PROJECTS' THEN
      /* DELETING A PROJECTS IMPORT */
      
      DELETE SIM_PROJECTS WHERE SPJ_IMPORT_SEQ = p_import_seq;
      COMMIT;
      
      
   ELSIF v_import_type = 'CHARACTERISTICS' THEN
      /* delete characteristics import */
      FOR v_acodes IN c_acodes(p_import_seq) LOOP
      
         DELETE SIM_ANALYSIS_CODES WHERE AC_SEQ = v_acodes.ac_seq;
     
         v_delete_cnt := v_delete_cnt + 1;
     
         IF v_delete_cnt > 100 THEN
            COMMIT;
            v_delete_cnt := 0;
         END IF;
     
      END LOOP;
     
   
    
   ELSE


      /* DELETING A RESULTS IMPORT */
      FOR v_field_activity IN c_field_activities(P_import_seq) LOOP

         DELETE sim_results WHERE rs_fa_seq = v_field_activity.fa_seq; 
         
         DELETE sim_activity_personnel WHERE sap_fa_seq = v_field_activity.fa_seq;
         
         DELETE sim_activity_projects WHERE sapj_fa_seq = v_field_activity.fa_seq;
    
         DELETE sim_field_activities WHERE fa_seq = v_field_activity.fa_seq;

         COMMIT;
          
         /* kms (6/11/01) - see if visit created by SIM, if so, see if any other activities
            are on the visit.  If there aren't and the visit is not still in STORET, delete
            the visit from the SIM_STATION_VISITS table. */
            
         OPEN c_get_visit_info(v_field_activity.FA_SSV_SEQ);
         FETCH c_get_visit_info INTO v_dummy_seq;
        -- CLOSE c_get_visit_info;
         
        -- IF v_created_by = 1 AND v_visit_is_nbr IS NULL AND v_visit_org_id IS NULL THEN       
           IF c_get_visit_info%NOTFOUND THEN
             
             /* kms - visit created by SIM and not associated with a visit in STORET, 
                see if there are any activities associated with it. */  
             OPEN c_check_visit_in_SIM(v_field_activity.FA_SSV_SEQ);
             FETCH c_check_visit_in_SIM INTO v_dummy_seq;
               IF c_check_visit_in_SIM%NOTFOUND THEN
               
                 /* kms - no activities associated with this visit, delete it. */
                 DELETE SIM_STATION_VISITS WHERE SSV_SEQ = v_field_activity.FA_SSV_SEQ;
                 
                 /* kms (6/11/01) - see if trip can be deleted.  Only do this when the station
                    visit is deleted since we know the possibility exists that there is nothing 
                    left on the trip. */
                 OPEN c_get_trip_info(v_field_activity.FA_STP_SEQ);
                 FETCH c_get_trip_info INTO v_dummy_seq;
                -- CLOSE c_get_trip_info;
  
                -- IF v_created_by = 1 AND v_trip_is_nbr IS NULL AND v_trip_org_id IS NULL THEN                
                   IF c_get_trip_info%NOTFOUND THEN
                   
                     /* kms - trip created by SIM and not associated with a trip in STORET,
                        see if there are any other station visits associated with it and 
                        delete if not */
                       OPEN c_check_trip_in_SIM(v_field_activity.FA_STP_SEQ);
                       FETCH c_check_trip_in_SIM INTO v_dummy_seq;
                       IF c_check_trip_in_SIM%NOTFOUND THEN
                       
                         /* kms - no station visits associated with the trip, delete. */
                         DELETE SIM_TRIPS WHERE STP_SEQ = v_field_activity.FA_STP_SEQ;
                         
                       END IF;
                     CLOSE c_check_trip_in_SIM;                     
                   END IF;
                 CLOSE c_get_trip_info;
               END IF;
             CLOSE c_check_visit_in_SIM;           
           END IF;
         CLOSE c_get_visit_info;
         COMMIT;
  
      END LOOP;
      
      COMMIT;

   END IF;
   
      
   /* clean out import lines table */
   /* DTW 09/27/01 Delete in batches of 100 to avoid rollback issues. */
   LOOP  
     OPEN c_check_import_line(p_import_seq);
     FETCH c_check_import_line INTO v_dummy_seq;
     IF c_check_import_line%NOTFOUND THEN
       CLOSE c_check_import_line;
       EXIT;   
     ELSE
       DELETE sim_imp_lines 
         WHERE SIPL_SIL_SEQ = p_import_seq
         AND rownum < 100;
       COMMIT;
       CLOSE c_check_import_line;
     END IF;  
   END LOOP;
  
       
   /* remove from import details */
   /* DTW 09/27/01 Delete in batches of 100 to avoid rollback issues. */
   LOOP  
     OPEN c_check_import_detail(p_import_seq);
     FETCH c_check_import_detail INTO v_dummy_seq;
     IF c_check_import_detail%NOTFOUND THEN
       CLOSE c_check_import_detail;
       EXIT;   
     ELSE
       DELETE sim_import_details 
         WHERE SID_SIL_SEQ = p_import_seq
         AND rownum < 100;
       COMMIT;
       CLOSE c_check_import_detail;
     END IF;  
   END LOOP;

   /* remove from import log */
   DELETE SIM_IMPORT_LOG WHERE SIL_SEQ =  p_import_seq;
   COMMIT;
   
   /*
   UPDATE SIM_IMPORT_LOG
      SET SIL_DELETED_ON = SYSDATE,
          SIL_DELETED_BY = USER
      WHERE SIL_SEQ = P_import_seq;
    COMMIT;
    */
    
END DELETE_IMPORT ;

/******************************************************************************************************/
FUNCTION GET_IMPORT_LOG_SEQ RETURN NUMBER IS                              

  CURSOR c_import_log_seq IS SELECT SIM_IMPORT_LOG_SEQ.NEXTVAL from DUAL;

  v_import_log_seq NUMBER;
  
BEGIN

   /* GET sequence for import log */
   OPEN c_import_log_seq;
   FETCH c_import_log_seq INTO v_import_log_seq;
   CLOSE c_import_log_seq;

   RETURN v_import_log_seq;

EXCEPTION

  WHEN OTHERS THEN
      SIM2_IMPORT_PKG.WRITE_ERROR (V_import_log_seq ,SQLERRM||' error in sim2_import_pkg.get_import_log_seq',null,null,null); 
   
END GET_IMPORT_LOG_SEQ;


PROCEDURE UPDATE_SIL_ROWS_READY(p_import_log_seq IN NUMBER, p_rows_ready_change IN NUMBER) IS

BEGIN

   UPDATE SIM_IMPORT_LOG 
   SET SIL_ROWS_READY = NVL(SIL_ROWS_READY,0) + NVL(p_rows_ready_change,0)
   WHERE SIL_SEQ = p_import_log_seq;

   COMMIT;

END UPDATE_SIL_ROWS_READY;

/******************************************************************************************************/
   
PROCEDURE UPDATE_SIL_ROWS_ERRORS(p_import_log_seq IN NUMBER, p_rows_errors_change IN NUMBER) IS

BEGIN

   UPDATE SIM_IMPORT_LOG 
   SET SIL_ROWS_ERRORS = NVL(SIL_ROWS_ERRORS,0) + NVL(p_rows_errors_change,0)
   WHERE SIL_SEQ = p_import_log_seq;

   COMMIT;


END UPDATE_SIL_ROWS_ERRORS;

/******************************************************************************************************/
   
PROCEDURE UPDATE_SIL_ROWS_IMPORTED(p_import_log_seq IN NUMBER, p_rows_imported_change IN NUMBER) IS

BEGIN

   UPDATE SIM_IMPORT_LOG 
   SET SIL_ROWS_IMPORTED = NVL(SIL_ROWS_IMPORTED,0) + NVL(p_rows_imported_change,0)
   WHERE SIL_SEQ = p_import_log_seq;

   COMMIT;

END UPDATE_SIL_ROWS_IMPORTED;
  


/******************************************************************************************************/

END SIM2_IMPORT_PKG;
/
